/*
 * Set maintenance mode for a linecard
 */
#include <stdio.h>
#include <unistd.h>

#include "libfma.h"
#include "lf_fms.h"

static void
usage()
{
  fprintf(stderr, "Set or clear maintenance mode for a linecard\n\n");
  fprintf(stderr, "Usage: fm_maint\n");
  fprintf(stderr, "  -h - print this help message\n");
  fprintf(stderr, "  -e <enclosure> - switch enclosure name\n");
  fprintf(stderr, "  -l <slot_no> - slot number of linecard to maintain\n");
  fprintf(stderr, "  [ -p <port_no> ] - port number to maintain\n");
  fprintf(stderr, "  -m { up | down } - set the state to up or down\n");
  fprintf(stderr, "  -s <fms_server> - specifies IP address of fms node\n");
  fprintf(stderr, "  -V - print version\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "If port number is not specified, then all ports will be affected.\n");
  exit(1);
}

int
main(
  int argc,
  char **argv)
{
  int c;
  char *fms_names;
  char *enc_name;
  int slot;
  int port_label;
  int fms;
  int up;
  int rc;
  extern char *optarg;

  lf_init();

  /* defaults */
  fms_names = NULL;
  enc_name = NULL;
  slot = -1;
  port_label = -1;
  up = -1;

  /*
   * Get command line args
   */
  while ((c = getopt(argc, argv, "hs:e:m:l:p:V")) != EOF) switch (c) {
  case 'h':
    usage();
    break;
  case 'm':
    if (strcmp(optarg, "up") == 0) {
      up = TRUE;
    } else if (strcmp(optarg, "down") == 0) {
      up = FALSE;
    } else {
      fprintf(stderr, "\"%s\" - bad value for -m\n", optarg);
      exit(1);
    }
    break;
  case 'l':
    slot = atoi(optarg);
    break;
  case 'p':
    port_label = atoi(optarg);
    break;
  case 'e':
    enc_name = optarg;
    break;
  case 's':
    fms_names = optarg;
    break;
  case 'V':
    printf("FMS version is %s\n", Lf_version);
    exit(0);
    break;
  }

  if (enc_name == NULL) {
    fprintf(stderr, "Need an enclosure name.\n");
    usage();
  }

  if (up == -1) {
    fprintf(stderr, "Need to specify up or down with -m\n");
    usage();
  }

  /*
   * Open connection to fms
   */
  fms = lf_fms_connect(fms_names);
  if (fms == -1) {
    fprintf(stderr, "Error connecting to ");
    if (fms_names == NULL) {
      fprintf(stderr, "default FMS\n");
    } else {
      fprintf(stderr, "FMS in (%s)\n", fms_names);
    }
    exit(1);
  }

  /*
   * Set maintenance mode
   */
  rc = lf_fms_switch_maintenance(fms, enc_name, slot, port_label, up);

  /* all done, shut down connection */
  lf_fms_disconnect(fms);

  if (rc == 0) {
    exit(0);
  } else {
    exit(1);
  }
}
